<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Blog;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Str;
use App\Services\ImageService;
use App\Traits\ImageHandler;

class BlogController extends Controller
{
    protected $imageService;

    public function __construct(ImageService $imageService)
    {
        $this->imageService = $imageService;
    }

    public function index()
    {
        $blogs = Blog::latest()->paginate(10);
        return view('admin.blog.index', compact('blogs'));
    }

    public function create()
    {
        return view('admin.blog.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'small_text' => 'nullable|string|max:255',
            'text' => 'nullable|string',
            'image' => 'nullable|image|max:10240', // 10MB
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'published_at' => 'nullable|date',
            'faq_schema' => 'nullable|string',
        ]);

        $data = $request->except(['image']);

        Carbon::setLocale('en');
        $date = Carbon::parse($request->published_at ?? now());
        $data['published_at'] = $date->locale('en')->translatedFormat('d F Y');
        $data['publish_at_en'] = $date->locale('en')->translatedFormat('d F Y');

        $data['slug'] = $this->generateUniqueSlug(Str::slug($request->title), 'slug');

        if ($request->hasFile('image')) {
            try {
                $data['image'] = $this->imageService->processImage(
                    $request->file('image'),
                    'uploads/blogs'
                );
            } catch (\Exception $e) {
                return back()->withInput()->withErrors(['image' => 'Image upload failed: ' . $e->getMessage()]);
            }
        }

        Blog::create($data);

        return redirect()->route('admin.blogs.index')->with('success', 'Blog created successfully.');
    }

    public function edit($id)
    {
        $blog = Blog::find($id);
        $row = $blog;
        return view('admin.blog.edit', compact('row'));
    }

    public function update(Request $request, $id)
    {
        $blog = Blog::find($id);
        $request->validate([
            'title' => 'required|string|max:255',
            'small_text' => 'nullable|string|max:255',
            'text' => 'nullable|string',
            'image' => 'nullable|image|max:10240',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'published_at' => 'nullable|date',
            'faq_schema' => 'nullable|string',
        ]);

        $data = $request->except(['image']);

        if ($request->hasFile('image')) {
             try {
                $this->imageService->deleteImage($blog->image);
                $data['image'] = $this->imageService->processImage(
                    $request->file('image'),
                    'uploads/blogs'
                );
            } catch (\Exception $e) {
                return back()->withInput()->withErrors(['image' => 'Image upload failed: ' . $e->getMessage()]);
            }
        }
        
      

        $blog->update($data);

        return redirect()->route('admin.blogs.index')->with('success', 'Blog updated successfully.');
    }

    public function destroy($id)
    {
        $blog = Blog::find($id);
        $blog->update(['is_active' => 0]);
        return redirect()->route('admin.blogs.index')->with('success', 'Blog deactivated successfully. To permanantly delete, contact admin.');
    }

    public function activate($id)
    {
        $blog = Blog::find($id);
        $blog->update(['is_active' => 1]);
        return redirect()->route('admin.blogs.index')->with('success', 'Blog activated successfully.');
    }

    private function generateUniqueSlug($slug, $column)
    {
       // Simplified slug generation
       $slug = Str::slug($slug);
       $count = 2;
       $original = $slug;
       while (Blog::where($column, $slug)->exists()) {
           $slug = "{$original}-{$count}";
           $count++;
       }
       return $slug;
    }
    
}
