<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Services\ImageService;
use App\Traits\ImageHandler;
use Illuminate\Http\Request;

class CategoryController extends Controller
{
    use ImageHandler;
    protected $imageService;

    public function __construct(ImageService $imageService)
    {
        $this->imageService = $imageService;
    }

    public function index()
    {
        $categories = Category::latest()->get();
        return view('admin.categories.index', compact('categories'));
    }

    public function create()
    {
        return view('admin.categories.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'meta_title' => 'required|string|max:255',
            'meta_description' => 'required|string',
            'page_schema' => 'required|string',
            'image' => 'nullable|image|max:10240',
            'is_active' => 'boolean',
        ]);
        $data = $request->all();

        $data['slug'] = $this->generateUniqueSlug(Str::slug($request->name), 'slug');

        if ($request->hasFile('image')) {
       
                $data['image'] = $this->processImage($request->file('image'), 'uploads/categories');
          
        }

        Category::create($data);

        return redirect()->route('admin.categories.index')->with('success', 'Category created successfully.');
    }

    public function edit($id)
    {
        $row = Category::find($id);
        return view('admin.categories.edit', compact('row'));
    }

    public function update(Request $request, $id)
    {
        $category = Category::find($id);
        if (!$category) {
            return redirect()->route('admin.categories.index')->with('error', 'Category not found.');
        }
        $request->validate([
            'name' => 'required|string|max:255',
            'meta_title' => 'required|string|max:255',
            'meta_description' => 'required|string',
            'page_schema' => 'required|string',
            'image' => 'nullable|image|max:10240',
            'is_active' => 'boolean',
        ]);

        $data = $request->all();

        if ($request->hasFile('image')) {
          
                $this->imageService->deleteImage($category->image);
                $data['image'] = $this->processImage($request->file('image'), 'uploads/categories');
           
        }

        $category->update($data);

        return redirect()->route('admin.categories.index')->with('success', 'Category updated successfully.');
    }

    public function destroy($id)
    {
        $category = Category::find($id);
        $category->update(['is_active' => 0]);
        return redirect()->route('admin.categories.index')->with('success', 'Category deleted successfully.');
    }

    public function activate($id)
    {
        $category = Category::find($id);
        $category->update(['is_active' => 1]);
        return redirect()->route('admin.categories.index')->with('success', 'Category activated successfully.');
    }

    private function generateUniqueSlug($slug, $column)
    {
        $slug = Str::slug($slug);
        $count = 2;
        $original = $slug;
        while (Category::where($column, $slug)->exists()) {
            $slug = "{$original}-{$count}";
            $count++;
        }
        return $slug;
    }
}
