<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Client;
use App\Services\ImageService;
use App\Traits\ImageHandler;
use Illuminate\Http\Request;

class ClientController extends Controller
{
    use ImageHandler;

    protected $imageService;

    public function __construct(ImageService $imageService)
    {
        $this->imageService = $imageService;
    }

    public function index()
    {
        $clients = Client::latest()->get();

        return view('admin.clients.index', compact('clients'));
    }

    public function create()
    {
        return view('admin.clients.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'client_url' => 'required|string|max:255',
            'image' => 'nullable|image|max:10240',
            'is_active' => 'boolean',
        ]);

        $data = $request->all();

        if ($request->hasFile('image')) {

            $data['image'] = $this->processImage($request->file('image'), 'uploads/clients');

        }

        Client::create($data);

        return redirect()->route('admin.clients.index')->with('success', 'Client created successfully.');
    }

    public function edit($id)
    {
        $row = Client::find($id);

        return view('admin.clients.edit', compact('row'));
    }

    public function update(Request $request, $id)
    {
        $client = Client::find($id);

        if (! $client) {
            return redirect()->route('admin.clients.index')->with('error', 'Client not found.');
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'client_url' => 'required|string|max:255',
            'image' => 'nullable|image|max:10240',
            'is_active' => 'boolean',
        ]);

        $data = $request->all();

        if ($request->hasFile('image')) {

            $this->imageService->deleteImage($client->image);
            $data['image'] = $this->processImage($request->file('image'), 'uploads/clients');

        }

        $client->update($data);

        return redirect()->route('admin.clients.index')->with('success', 'Client updated successfully.');
    }

    public function destroy($id)
    {
        $client = Client::find($id);
        $client->update(['is_active' => 0]);

        return redirect()->route('admin.clients.index')->with('success', 'Client deleted successfully.');
    }

    public function activate($id)
    {
        $client = Client::find($id);
        $client->update(['is_active' => 1]);

        return redirect()->route('admin.clients.index')->with('success', 'Client activated successfully.');
    }
}
