<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Project;
use App\Services\ImageService;
use App\Traits\ImageHandler;
use Illuminate\Http\Request;

class ProjectController extends Controller
{
    use ImageHandler;
    protected $imageService;

    public function __construct(ImageService $imageService)
    {
        $this->imageService = $imageService;
    }

    public function index()
    {
        $projects = Project::latest()->get();
        return view('admin.projects.index', compact('projects'));
    }

    public function create()
    {
        $categories = Category::latest()->get();
        return view('admin.projects.create', compact('categories'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'category_id' => 'required|exists:categories,id',
            'title' => 'required|string|max:255',
            'slug' => 'required|string|max:255',
            'description' => 'nullable|string',
            'project_url' => 'required|string|max:255',
            'meta_title' => 'required|string|max:255',
            'meta_description' => 'nullable|string',
            'page_schema' => 'nullable|string',
            'image' => 'nullable|image|max:10240',
            'is_active' => 'boolean',
        ]);
        $data = $request->all();

        $data['slug'] = $this->generateArabicSlug($request->title);


        if ($request->hasFile('image')) {
       
                $data['image'] = $this->processImage($request->file('image'), 'uploads/projects');
          
        }

        Project::create($data);

        return redirect()->route('admin.projects.index')->with('success', 'Project created successfully.');
    }

    public function edit($id)
    {
        $row = Project::find($id);
        $categories = Category::latest()->get();
        return view('admin.projects.edit', compact('row', 'categories'));
    }

    public function update(Request $request, $id)
    {
        $project = Project::find($id);
        if (!$project) {
            return redirect()->route('admin.projects.index')->with('error', 'Project not found.');
        }
        $request->validate([
            'category_id' => 'required|exists:categories,id',
            'title' => 'required|string|max:255',
            'slug' => 'required|string|max:255',
            'description' => 'nullable|string',
            'project_url' => 'required|string|max:255',
            'meta_title' => 'required|string|max:255',
            'meta_description' => 'nullable|string',
            'page_schema' => 'nullable|string',
            'meta_title' => 'required|string|max:255',
            'meta_description' => 'nullable|string',
            'page_schema' => 'nullable|string',
            'image' => 'nullable|image|max:10240',
            'is_active' => 'boolean',
        ]);

        $data = $request->all();

        if ($request->hasFile('image')) {
          
                $this->imageService->deleteImage($project->image);
                $data['image'] = $this->processImage($request->file('image'), 'projects');
           
        }

        $project->update($data);

        return redirect()->route('admin.projects.index')->with('success', 'Project updated successfully.');
    }

    public function destroy($id)
    {
        $project = Project::find($id);
        $project->update(['is_active' => 0]);
        return redirect()->route('admin.projects.index')->with('success', 'Project deleted successfully.');
    }

    public function activate($id)
    {
        $project = Project::find($id);
        $project->update(['is_active' => 1]);
        return redirect()->route('admin.projects.index')->with('success', 'Project activated successfully.');
    }

    private function generateUniqueSlug($slug, $column)
    {
        $slug = Str::slug($slug);
        $count = 2;
        $original = $slug;
        while (Category::where($column, $slug)->exists()) {
            $slug = "{$original}-{$count}";
            $count++;
        }
        return $slug;
    }

    private function generateArabicSlug($title)
    {
        // تحويل المسافات والفواصل لشرطة
        $slug = preg_replace('/[^\p{L}\p{N}]+/u', '-', $title);

        // إزالة التكرار
        $slug = preg_replace('/-+/', '-', $slug);

        // حذف الشرطة من البداية والنهاية
        $slug = trim($slug, '-');

        return mb_strtolower($slug, 'UTF-8');
    }
}
