<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SocialMedia;
use App\Services\ImageService;
use App\Traits\ImageHandler;
use Illuminate\Http\Request;

class SocialMediaController extends Controller
{
    use ImageHandler;
    protected $imageService;

    public function __construct(ImageService $imageService)
    {
        $this->imageService = $imageService;
    }

    public function index()
    {
        $socialMedias = SocialMedia::latest()->get();
        return view('admin.social_medias.index', compact('socialMedias'));
    }

    public function create()
    {
        return view('admin.social_medias.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'link' => 'required|url',
            'image' => 'nullable|image|max:10240',
            'is_active' => 'boolean',
        ]);

        $data = $request->all();

        if ($request->hasFile('image')) {
            try {
                $data['image'] = $this->processImage(
                    $request->file('image'),
                    'uploads/social-medias'
                );
            } catch (\Exception $e) {
                return back()->withInput()->withErrors(['image' => 'Image upload failed: ' . $e->getMessage()]);
            }
        }

        SocialMedia::create($data);

        return redirect()->route('admin.social_medias.index')->with('success', 'Social Media created successfully.');
    }

    public function edit($id)
    {
        $socialMedia = SocialMedia::find($id);
        $row = $socialMedia;
        return view('admin.social_medias.edit', compact('row'));
    }
   

    public function update(Request $request, $id)
    {
        $socialMedia = SocialMedia::find($id);
        $request->validate([
            'name' => 'required|string|max:255',
            'link' => 'required|url',
            'image' => 'nullable|image|max:10240',
            'is_active' => 'boolean',
        ]);

        $data = $request->all();

        if ($request->hasFile('image')) {
            try {
                $this->imageService->deleteImage($socialMedia->image);
                $data['image'] = $this->processImage(
                    $request->file('image'),
                    'uploads/social-medias'
                );
            } catch (\Exception $e) {
                return back()->withInput()->withErrors(['image' => 'Image upload failed: ' . $e->getMessage()]);
            }
        }

        $socialMedia->update($data);

        return redirect()->route('admin.social_medias.index')->with('success', 'Social Media updated successfully.');
    }

    public function destroy($id)
    {
        $socialMedia = SocialMedia::find($id);
        $socialMedia->update(['is_active' => 0]);
        return redirect()->route('admin.social_medias.index')->with('success', 'Social Media deleted successfully.');
    }

    public function activate($id)
    {
        $socialMedia = SocialMedia::find($id);
        $socialMedia->update(['is_active' => 1]);
        return redirect()->route('admin.social_medias.index')->with('success', 'Social Media activated successfully.');
    }
}
