<?php

namespace App\Services;

use Intervention\Image\ImageManager;
use Intervention\Image\Drivers\Gd\Driver;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;

class ImageService
{
    protected $manager;

    public function __construct()
    {
        $this->manager = new ImageManager(new Driver());
    }

    /**
     * Convert and save image to WebP format
     * Falls back to original format if WebP conversion fails
     */
    public function convertToWebP(UploadedFile $file, string $path, int $quality = 85): string
    {
        // Validate file exists and is readable
        if (!$file->isValid()) {
            throw new \Exception('Uploaded file is not valid');
        }

        $filePath = $file->getRealPath();
        if (!$filePath || !file_exists($filePath)) {
            throw new \Exception('File path is invalid or file does not exist');
        }

        // Ensure directory exists using Storage
        if (!Storage::disk('public')->exists($path)) {
            Storage::disk('public')->makeDirectory($path);
        }

        try {
            // Read the image
            $image = $this->manager->read($filePath);
            
            // Resize if needed (optional - you can adjust max dimensions)
            
            // Generate unique filename
            $filename = uniqid() . '.webp';
            $webpPath = $path . '/' . $filename;
            
            // Convert to WebP and get the encoded data
            $encoded = $image->toWebp($quality);
            
            // Get the binary data from the encoded image
            $webpData = $encoded->toString();
            
            // Save directly using Storage
            Storage::disk('public')->put($webpPath, $webpData);
            
            // Verify file was saved
            if (!Storage::disk('public')->exists($webpPath)) {
                throw new \Exception('Failed to save WebP file to storage');
            }
            
            return $webpPath;
        } catch (\Exception $e) {
            // Log the error but don't fail - fall back to original format
          
            
            // Fallback: save original file
            $extension = $file->getClientOriginalExtension() ?: 'jpg';
            $filename = uniqid() . '.' . $extension;
            $originalPath = $path . '/' . $filename;
            
            Storage::disk('public')->putFileAs($path, $file, $filename);
            
            return $originalPath;
        }
    }

    /**
     * Delete old image if exists
     */
    public function deleteImage(?string $path): void
    {
        if ($path && Storage::disk('public')->exists($path)) {
            Storage::disk('public')->delete($path);
        }
    }
}

