<?php

$modules = [
    'courses' => [
        'model' => 'Course',
        'route_prefix' => 'admin.courses',
        'fields' => [
            ['name' => 'title', 'type' => 'text', 'label' => 'Title', 'required' => true],
            ['name' => 'category_id', 'type' => 'select', 'label' => 'Category', 'options_variable' => 'categories', 'option_label' => 'name', 'required' => true],
            ['name' => 'image', 'type' => 'image', 'label' => 'Image'],
            ['name' => 'description', 'type' => 'textarea', 'label' => 'Description', 'required' => true],
            ['name' => 'lesson_number', 'type' => 'number', 'label' => 'Lesson Number'],
            ['name' => 'duration', 'type' => 'text', 'label' => 'Duration'],
            ['name' => 'languages', 'type' => 'text', 'label' => 'Languages'],
            ['name' => 'is_active', 'type' => 'checkbox', 'label' => 'Active', 'default' => true],
        ]
    ],
    'course_categories' => [
        'folder' => 'course_categories', // Blade folder name
        'model' => 'CourseCategory',
        'route_prefix' => 'admin.course-categories',
        'fields' => [
            ['name' => 'name', 'type' => 'text', 'label' => 'Name', 'required' => true],
            ['name' => 'image', 'type' => 'image', 'label' => 'Image'],
        ]
    ],
    'features' => [
        'model' => 'Feature',
        'route_prefix' => 'admin.features',
        'fields' => [
            ['name' => 'title', 'type' => 'text', 'label' => 'Title', 'required' => true],
            ['name' => 'text', 'type' => 'textarea', 'label' => 'Text', 'required' => true],
            ['name' => 'image', 'type' => 'image', 'label' => 'Image'],
            ['name' => 'is_active', 'type' => 'checkbox', 'label' => 'Active', 'default' => true],
        ]
    ],
    'feature_sections' => [
        'folder' => 'feature_sections',
        'model' => 'FeatureSection',
        'route_prefix' => 'admin.feature-sections',
        'fields' => [
            ['name' => 'title', 'type' => 'text', 'label' => 'Title', 'required' => true],
            ['name' => 'image', 'type' => 'image', 'label' => 'Image'],
        ]
    ],
    'lessons' => [
        'model' => 'Lesson',
        'route_prefix' => 'admin.lessons',
        'fields' => [
            ['name' => 'title', 'type' => 'text', 'label' => 'Title', 'required' => true],
            ['name' => 'course_id', 'type' => 'select', 'label' => 'Course', 'options_variable' => 'courses', 'option_label' => 'title', 'required' => true],
            ['name' => 'lesson_category_id', 'type' => 'select', 'label' => 'Category', 'options_variable' => 'categories', 'option_label' => 'title', 'required' => true],
            ['name' => 'video_url', 'type' => 'url', 'label' => 'Video URL', 'required' => true],
            ['name' => 'duration', 'type' => 'text', 'label' => 'Duration'],
            ['name' => 'is_active', 'type' => 'checkbox', 'label' => 'Active', 'default' => true],
        ]
    ],
    'lesson_categories' => [
        'folder' => 'lesson_categories',
        'model' => 'LessonCategory',
        'route_prefix' => 'admin.lesson-categories',
        'fields' => [
            ['name' => 'title', 'type' => 'text', 'label' => 'Title', 'required' => true],
            ['name' => 'course_id', 'type' => 'select', 'label' => 'Course', 'options_variable' => 'courses', 'option_label' => 'title', 'required' => true],
            ['name' => 'is_active', 'type' => 'checkbox', 'label' => 'Active', 'default' => true],
        ]
    ],
    'social_medias' => [
        'folder' => 'social_medias',
        'model' => 'SocialMedia',
        'route_prefix' => 'admin.social-medias',
        'fields' => [
            ['name' => 'name', 'type' => 'text', 'label' => 'Name', 'required' => true],
            ['name' => 'link', 'type' => 'url', 'label' => 'Link', 'required' => true],
            ['name' => 'image', 'type' => 'image', 'label' => 'Icon/Image'],
            ['name' => 'is_active', 'type' => 'checkbox', 'label' => 'Active', 'default' => true],
        ]
    ],
    'testimonials' => [
        'model' => 'Testimonial',
        'route_prefix' => 'admin.testimonials',
        'fields' => [
            ['name' => 'name', 'type' => 'text', 'label' => 'Name', 'required' => true],
            ['name' => 'title', 'type' => 'text', 'label' => 'Title/Role'],
            ['name' => 'description', 'type' => 'textarea', 'label' => 'Description', 'required' => true],
            ['name' => 'image', 'type' => 'image', 'label' => 'Image'],
        ]
    ],
    'users' => [
        'model' => 'User',
        'route_prefix' => 'admin.users',
        'fields' => [
            ['name' => 'name', 'type' => 'text', 'label' => 'Name', 'required' => true],
            ['name' => 'email', 'type' => 'email', 'label' => 'Email', 'required' => true],
            ['name' => 'password', 'type' => 'password', 'label' => 'Password (Leave blank to keep current)'],
            ['name' => 'phone', 'type' => 'text', 'label' => 'Phone'],
            ['name' => 'role', 'type' => 'text', 'label' => 'Role'],
            ['name' => 'is_admin', 'type' => 'checkbox', 'label' => 'Is Admin', 'default' => false],
        ]
    ],
    'user_course_progress' => [
        'folder' => 'user_course_progress',
        'model' => 'UserCourseProgress',
        'route_prefix' => 'admin.user-course-progress',
        'fields' => [], // Mostly index view, keeping edit simple or skipped for now as it's progress data
        'skip_create_edit' => true
    ],
    'contact_forms' => [
        'folder' => 'contact_forms',
        'model' => 'ContactForm',
        'route_prefix' => 'admin.contact-forms',
        'fields' => [
            ['name' => 'name', 'type' => 'text', 'label' => 'Name'],
            ['name' => 'email', 'type' => 'text', 'label' => 'Email'],
            ['name' => 'phone', 'type' => 'text', 'label' => 'Phone'],
            ['name' => 'message', 'type' => 'textarea', 'label' => 'Message'],
            ['name' => 'created_at', 'type' => 'text', 'label' => 'Date'],
            ['name' => 'is_read', 'type' => 'checkbox', 'label' => 'Read', 'default' => false],
        ],
        'skip_create' => true // Usually no create for contact forms
    ],
    'course_forms' => [
        'folder' => 'course_forms',
        'model' => 'CourseForm',
        'route_prefix' => 'admin.course-forms',
        'fields' => [
            ['name' => 'name', 'type' => 'text', 'label' => 'Name'],
            ['name' => 'email', 'type' => 'text', 'label' => 'Email'],
            ['name' => 'phone', 'type' => 'text', 'label' => 'Phone'],
            ['name' => 'course_id', 'type' => 'select', 'label' => 'Course', 'options_variable' => 'courses', 'option_label' => 'title', 'required' => false],
            ['name' => 'is_read', 'type' => 'checkbox', 'label' => 'Read', 'default' => false],
        ],
        'skip_create' => true
    ],
    'blogs' => [
        'folder' => 'blog', // Overwrite existing folder
        'model' => 'Blog',
        'route_prefix' => 'admin.blogs',
        'fields' => [
            ['name' => 'title', 'type' => 'text', 'label' => 'Title', 'required' => true],
            ['name' => 'small_text', 'type' => 'textarea', 'label' => 'Small Text'],
            ['name' => 'text', 'type' => 'textarea', 'label' => 'Content'],
            ['name' => 'main_image', 'type' => 'image', 'label' => 'Main Image'],
            ['name' => 'meta_title', 'type' => 'text', 'label' => 'Meta Title'],
            ['name' => 'meta_description', 'type' => 'textarea', 'label' => 'Meta Description'],
            ['name' => 'created_at', 'type' => 'text', 'label' => 'Published Date'],
            ['name' => 'is_active', 'type' => 'checkbox', 'label' => 'Active', 'default' => true],
        ]
    ],
    'privacy_policies' => [
        'folder' => 'privacy_policy', // Overwrite
        'model' => 'PrivacyPolicy',
        'route_prefix' => 'admin.privacy-policies',
        'fields' => [
            ['name' => 'title', 'type' => 'text', 'label' => 'Title', 'required' => true],
            ['name' => 'text', 'type' => 'textarea', 'label' => 'Text', 'required' => true],
        ],
        'skip_create' => true // Usually singleton or limited
    ],
    'page_informations' => [
        'folder' => 'page_information', // Overwrite
        'model' => 'PageInformation',
        'route_prefix' => 'admin.page-informations',
        'fields' => [
            ['name' => 'title', 'type' => 'text', 'label' => 'Title', 'required' => true],
            ['name' => 'text', 'type' => 'textarea', 'label' => 'Text'],
            ['name' => 'main_image', 'type' => 'image', 'label' => 'Main Image'],
        ],
        'skip_create' => true
    ],
    'contact_information' => [
        'folder' => 'contact_information',
        'model' => 'ContactInformation',
        'route_prefix' => 'admin.contact-informations',
        'fields' => [
             ['name' => 'email', 'type' => 'text', 'label' => 'Email'],
             ['name' => 'phone', 'type' => 'text', 'label' => 'Phone'],
             ['name' => 'address', 'type' => 'text', 'label' => 'Address'],
             ['name' => 'location', 'type' => 'textarea', 'label' => 'Location (Map Embed)'],
             ['name' => 'footer_text', 'type' => 'textarea', 'label' => 'Footer Text'],
        ],
        'skip_create' => true
    ],
];

function generateBladeFiles($moduleName, $config) {
    $folderName = $config['folder'] ?? $moduleName;
    $basePath = __DIR__ . '/../resources/views/admin/' . $folderName;
    
    if (!file_exists($basePath)) {
        mkdir($basePath, 0755, true);
    }
    
    $modelName = $config['model'];
    $routePrefix = $config['route_prefix'];
    $fields = $config['fields'];
    $variableName = \Illuminate\Support\Str::camel($modelName); // e.g., courseCategory
    $pluralVariable = \Illuminate\Support\Str::plural($variableName); // e.g., courseCategories (Wait, Str helper not avail here easily without booting framework)

    // Quick helper for plural/camel
    $camelCase = lcfirst($modelName);
    $pluralVar = $moduleName == 'course_categories' || $moduleName == 'lesson_categories' || $moduleName == 'social_medias' || $moduleName == 'feature_sections' 
        ? lcfirst(\Illuminate\Support\Str::camel($moduleName)) 
        : $moduleName; // Simplified
    
    // 1. Index View
    $indexContent = generateIndexContent($modelName, $routePrefix, $fields, $pluralVar, $config);
    file_put_contents("$basePath/index.blade.php", $indexContent);
    echo "Created $basePath/index.blade.php\n";

    if (isset($config['skip_create_edit']) && $config['skip_create_edit']) {
        return;
    }

    // 2. Create View
    $createContent = generateCreateContent($modelName, $routePrefix, $fields, $variableName);
    file_put_contents("$basePath/create.blade.php", $createContent);
    echo "Created $basePath/create.blade.php\n";

    // 3. Edit View
    $editContent = generateEditContent($modelName, $routePrefix, $fields, $variableName);
    file_put_contents("$basePath/edit.blade.php", $editContent);
    echo "Created $basePath/edit.blade.php\n";
}

function generateIndexContent($model, $route, $fields, $pluralVar, $config) {
    // Generate Table Headers
    $headers = "";
    $rows = "";
    
    foreach ($fields as $field) {
        if ($field['type'] === 'password' || $field['type'] === 'textarea' && $field['name'] !== 'description') continue; // Skip detailed text in index
        $label = $field['label'];
        $headers .= "                    <th class=\"px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase\">$label</th>\n";
        
        $fieldName = $field['name'];
        $rowContent = "";
        
        if ($field['type'] === 'image') {
            $rowContent = "
                            @if(\$item->$fieldName)
                                <img src=\"{{ Storage::url(\$item->$fieldName) }}\" alt=\"Image\" class=\"h-10 w-10 object-cover rounded\">
                            @else
                                <span class=\"text-gray-400\">-</span>
                            @endif";
        } elseif ($field['type'] === 'checkbox') {
             $rowContent = "
                            @if(\$item->$fieldName)
                                <span class=\"px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200\">Yes</span>
                            @else
                                <span class=\"px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-300\">No</span>
                            @endif";
        } elseif ($field['type'] === 'select') {
             // Try to guess relation name: category_id -> category->name
             $relation = str_replace('_id', '', $fieldName);
             $optionLabel = $field['option_label'] ?? 'name';
             // camel case relation
             $relation = lcfirst(str_replace(' ', '', ucwords(str_replace('_', ' ', $relation)))); 
             $rowContent = "{{ \$item->$relation ? \$item->$relation->$optionLabel : '-' }}";
        } else {
             $rowContent = "{{ Str::limit(\$item->$fieldName, 50) }}";
        }
        
        $rows .= "                        <td class=\"px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-300\">$rowContent</td>\n";
    }

    return <<<BLADE
<x-admin-layout>
    <x-slot name="header">
        $model Management
    </x-slot>

    <div class="mb-4 flex justify-between items-center">
        <h2 class="text-xl font-semibold text-gray-900 dark:text-white">All $model Plural</h2>
        <a href="{{ route('$route.create') }}" class="bg-blue-500 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded">
            Add New $model
        </a>
    </div>

    @if(session('success'))
        <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
            {{ session('success') }}
        </div>
    @endif

    <div class="bg-white dark:bg-gray-800 rounded-lg shadow overflow-hidden">
        <table class="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
            <thead class="bg-gray-50 dark:bg-gray-900">
                <tr>
$headers
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Actions</th>
                </tr>
            </thead>
            <tbody class="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                @forelse(\$$pluralVar as \$item)
                    <tr>
$rows
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium space-x-2">
                            <a href="{{ route('$route.edit', \$item) }}" class="text-blue-600 hover:text-blue-900 dark:text-blue-400">Edit</a>
                            <form action="{{ route('$route.destroy', \$item) }}" method="POST" class="inline">
                                @csrf
                                @method('DELETE')
                                <button type="submit" class="text-red-600 hover:text-red-900 dark:text-red-400" onclick="return confirm('Are you sure you want to delete this?')">Delete</button>
                            </form>
                        </td>
                    </tr>
                @empty
                    <tr>
                        <td colspan="100" class="px-6 py-4 text-center text-sm text-gray-500 dark:text-gray-400">No records found.</td>
                    </tr>
                @endforelse
            </tbody>
        </table>
        <div class="p-4">
            {{ \$$pluralVar->links() }}
        </div>
    </div>
</x-admin-layout>
BLADE;
}

function generateCreateContent($model, $route, $fields, $variableName) {
    $inputs = "";
    
    foreach ($fields as $field) {
        $name = $field['name'];
        $label = $field['label'];
        $type = $field['type'];
        $required = isset($field['required']) && $field['required'] ? 'required' : '';
        $reqLabel = isset($field['required']) && $field['required'] ? '*' : '';

        if ($type === 'image') {
            $inputs .= "            <x-image-upload name=\"$name\" label=\"$label\" />\n";
        } elseif ($type === 'textarea') {
            $inputs .= "
            <div class=\"mb-4\">
                <label for=\"$name\" class=\"block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2\">$label $reqLabel</label>
                <textarea name=\"$name\" id=\"$name\" rows=\"4\" $required
                    class=\"w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white\">{{ old('$name') }}</textarea>
                @error('$name')
                    <p class=\"mt-1 text-sm text-red-600\">{{ \$message }}</p>
                @enderror
            </div>\n";
        } elseif ($type === 'checkbox') {
             $checked = isset($field['default']) && $field['default'] ? "checked" : "";
             $inputs .= "
            <div class=\"mb-4\">
                <label class=\"flex items-center\">
                    <input type=\"hidden\" name=\"$name\" value=\"0\">
                    <input type=\"checkbox\" name=\"$name\" value=\"1\" {{ old('$name', 1) ? 'checked' : '' }}
                        class=\"rounded border-gray-300 text-blue-600 shadow-sm focus:border-blue-300 focus:ring focus:ring-blue-200 focus:ring-opacity-50\">
                    <span class=\"ml-2 text-sm text-gray-700 dark:text-gray-300\">$label</span>
                </label>
            </div>\n";
        } elseif ($type === 'select') {
             $optionsVar = $field['options_variable'];
             $optionLabel = $field['option_label'];
             $inputs .= "
            <div class=\"mb-4\">
                <label for=\"$name\" class=\"block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2\">$label $reqLabel</label>
                <select name=\"$name\" id=\"$name\" $required class=\"w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white\">
                    <option value=\"\">Select $label</option>
                    @foreach(\$$optionsVar as \$option)
                        <option value=\"{{ \$option->id }}\" {{ old('$name') == \$option->id ? 'selected' : '' }}>{{ \$option->$optionLabel }}</option>
                    @endforeach
                </select>
                @error('$name')
                    <p class=\"mt-1 text-sm text-red-600\">{{ \$message }}</p>
                @enderror
            </div>\n";
        } elseif ($type === 'password') {
             $inputs .= "
            <div class=\"mb-4\">
                <label for=\"$name\" class=\"block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2\">$label $reqLabel</label>
                <input type=\"password\" name=\"$name\" id=\"$name\" $required
                    class=\"w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white\">
                @error('$name')
                    <p class=\"mt-1 text-sm text-red-600\">{{ \$message }}</p>
                @enderror
            </div>
            <div class=\"mb-4\">
                <label for=\"password_confirmation\" class=\"block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2\">Confirm Password</label>
                <input type=\"password\" name=\"password_confirmation\" id=\"password_confirmation\"
                    class=\"w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white\">
            </div>\n";
        } else {
             $inputs .= "
            <div class=\"mb-4\">
                <label for=\"$name\" class=\"block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2\">$label $reqLabel</label>
                <input type=\"$type\" name=\"$name\" id=\"$name\" value=\"{{ old('$name') }}\" $required
                    class=\"w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white\">
                @error('$name')
                    <p class=\"mt-1 text-sm text-red-600\">{{ \$message }}</p>
                @enderror
            </div>\n";
        }
    }

    return <<<BLADE
<x-admin-layout>
    <x-slot name="header">
        Create New $model
    </x-slot>

    @if(\$errors->any())
        <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
            <ul class="list-disc list-inside">
                @foreach(\$errors->all() as \$error)
                    <li>{{ \$error }}</li>
                @endforeach
            </ul>
        </div>
    @endif

    <div class="max-w-2xl">
        <form action="{{ route('$route.store') }}" method="POST" enctype="multipart/form-data" class="bg-white dark:bg-gray-800 rounded-lg shadow p-6">
            @csrf

$inputs

            <div class="flex space-x-4">
                <button type="submit" class="bg-blue-500 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded">
                    Create $model
                </button>
                <a href="{{ route('$route.index') }}" class="bg-gray-500 hover:bg-gray-700 text-white font-bold py-2 px-4 rounded">
                    Cancel
                </a>
            </div>
        </form>
    </div>
</x-admin-layout>
BLADE;
}

function generateEditContent($model, $route, $fields, $variableName) {
       $inputs = "";
    
    foreach ($fields as $field) {
        $name = $field['name'];
        $label = $field['label'];
        $type = $field['type'];
        $required = isset($field['required']) && $field['required'] ? 'required' : '';
        $reqLabel = isset($field['required']) && $field['required'] ? '*' : '';

        // For Edit, Password required is ignored usually or handled differently
        if ($type === 'password') $required = ''; 

        if ($type === 'image') {
            $inputs .= "            <x-image-upload name=\"$name\" label=\"$label\" :value=\"\$item->$name\" />\n";
        } elseif ($type === 'textarea') {
            $inputs .= "
            <div class=\"mb-4\">
                <label for=\"$name\" class=\"block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2\">$label $reqLabel</label>
                <textarea name=\"$name\" id=\"$name\" rows=\"4\" $required
                    class=\"w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white\">{{ old('$name', \$item->$name) }}</textarea>
                @error('$name')
                    <p class=\"mt-1 text-sm text-red-600\">{{ \$message }}</p>
                @enderror
            </div>\n";
        } elseif ($type === 'checkbox') {
             $inputs .= "
            <div class=\"mb-4\">
                <label class=\"flex items-center\">
                    <input type=\"hidden\" name=\"$name\" value=\"0\">
                    <input type=\"checkbox\" name=\"$name\" value=\"1\" {{ old('$name', \$item->$name) ? 'checked' : '' }}
                        class=\"rounded border-gray-300 text-blue-600 shadow-sm focus:border-blue-300 focus:ring focus:ring-blue-200 focus:ring-opacity-50\">
                    <span class=\"ml-2 text-sm text-gray-700 dark:text-gray-300\">$label</span>
                </label>
            </div>\n";
         } elseif ($type === 'select') {
             $optionsVar = $field['options_variable'];
             $optionLabel = $field['option_label'];
             $inputs .= "
            <div class=\"mb-4\">
                <label for=\"$name\" class=\"block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2\">$label $reqLabel</label>
                <select name=\"$name\" id=\"$name\" $required class=\"w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white\">
                    <option value=\"\">Select $label</option>
                    @foreach(\$$optionsVar as \$option)
                        <option value=\"{{ \$option->id }}\" {{ old('$name', \$item->$name) == \$option->id ? 'selected' : '' }}>{{ \$option->$optionLabel }}</option>
                    @endforeach
                </select>
                @error('$name')
                    <p class=\"mt-1 text-sm text-red-600\">{{ \$message }}</p>
                @enderror
            </div>\n";
        } elseif ($type === 'password') {
             $inputs .= "
            <div class=\"mb-4\">
                <label for=\"$name\" class=\"block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2\">$label (Leave blank to keep current)</label>
                <input type=\"password\" name=\"$name\" id=\"$name\"
                    class=\"w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white\">
                @error('$name')
                    <p class=\"mt-1 text-sm text-red-600\">{{ \$message }}</p>
                @enderror
            </div>
            <div class=\"mb-4\">
                <label for=\"password_confirmation\" class=\"block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2\">Confirm Password</label>
                <input type=\"password\" name=\"password_confirmation\" id=\"password_confirmation\"
                    class=\"w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white\">
            </div>\n";
        } else {
             $inputs .= "
            <div class=\"mb-4\">
                <label for=\"$name\" class=\"block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2\">$label $reqLabel</label>
                <input type=\"$type\" name=\"$name\" id=\"$name\" value=\"{{ old('$name', \$item->$name) }}\" $required
                    class=\"w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white\">
                @error('$name')
                    <p class=\"mt-1 text-sm text-red-600\">{{ \$message }}</p>
                @enderror
            </div>\n";
        }
    }
    
    // Determine variable name in view, usually passed as \$course, \$feature etc.
    // In resource controller defaults, it might be same as param name.
    // I used `compact('course')` in controllers.
    // So \$item should be aliased to \$$variableName at top of view or just use \$$variableName
    // To be generic, I will use \$item in the loop above but I need to assign it.
    // Better: just use \$$variableName directly.
    
    // REPLACE \$item with \$$variableName in inputs
    $inputs = str_replace('\$item', "\$$variableName", $inputs);

    return <<<BLADE
<x-admin-layout>
    <x-slot name="header">
        Edit $model
    </x-slot>

    @if(\$errors->any())
        <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
            <ul class="list-disc list-inside">
                @foreach(\$errors->all() as \$error)
                    <li>{{ \$error }}</li>
                @endforeach
            </ul>
        </div>
    @endif

    <div class="max-w-2xl">
        <form action="{{ route('$route.update', \$$variableName) }}" method="POST" enctype="multipart/form-data" class="bg-white dark:bg-gray-800 rounded-lg shadow p-6">
            @csrf
            @method('PUT')

$inputs

            <div class="flex space-x-4">
                <button type="submit" class="bg-blue-500 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded">
                    Update $model
                </button>
                <a href="{{ route('$route.index') }}" class="bg-gray-500 hover:bg-gray-700 text-white font-bold py-2 px-4 rounded">
                    Cancel
                </a>
            </div>
        </form>
    </div>
</x-admin-layout>
BLADE;
}

// Support for Laravel helpers when running standalone
if (!function_exists('app')) {
    require __DIR__ . '/../vendor/autoload.php';
    // Minimal mock if needed, but we mostly use Str
}

foreach ($modules as $name => $config) {
    echo "Generating views for $name...\n";
    generateBladeFiles($name, $config);
}
